﻿Imports System.Drawing
Imports System.Drawing.Imaging
Imports System.IO
Imports System.Text
Imports System.Windows.Threading
Imports System.Windows.Interop

Class MainWindow

    Dim timerREC As New DispatcherTimer()
    Dim timerFPS As New DispatcherTimer()

    Private Sub Window_Loaded(sender As Object, e As RoutedEventArgs)

        ' Make the log file path the same as the process exe file path but different extension.
        Dim exePath As String = Process.GetCurrentProcess().MainModule.FileName
        Dim logPath As String = Path.ChangeExtension(exePath, ".txt")

        ' Create a log file that Is helpful for debugging.
        ' It can be called before calling ScnLib_Initialize().
        ZDSoft.SDK.ScnLib_SetLogPathW(logPath, True)

        ' Initialize the SDK.
        ' It must be called before calling most of the other APIs of the SDK.
        ' Usually it's called only once at the start of the program.
        ZDSoft.SDK.ScnLib_InitializeW("")

        ' Make the screen capture region frame visible. It's invisible by default.
        ' If you don't like the style of the SDK built-in screen capture region frame, you may implement your own.
        ZDSoft.SDK.ScnLib_ShowCaptureRegionFrame(True)

        ' We prefer 1080p webcam resolution. It's 320x240 by default.
        ' If your webcam doesn't support the resolution you set, the SDK will choose the closest resolution automatically.
        ZDSoft.SDK.ScnLib_SetWebcamResolution(1920, 1080)

        ' Prepare a timer for checking the recording status and updating recording time.
        AddHandler timerREC.Tick, AddressOf On_Timer_REC
        timerREC.Interval = TimeSpan.FromSeconds(0.5)

        ' Setup a timer to update the game FPS overlay for game capture mode
        AddHandler timerFPS.Tick, AddressOf On_Timer_FPS
        timerFPS.Interval = TimeSpan.FromSeconds(1)
        timerFPS.Start()

        Update_UI()

    End Sub

    Private Sub Window_Closed(sender As Object, e As EventArgs)

        ' Uninitialize the SDK.
        ' It must be called before exiting the process or there might be resource leak.
        ' Usually it's called only once at the end of the program.
        ZDSoft.SDK.ScnLib_Uninitialize()

    End Sub

    Private Sub On_Timer_REC(sender As Object, e As EventArgs)

        ' Since the SDK doesn't have any callback mechanism, 
        ' you need to check the recording status periodically in a timerREC.
        ' If it's recording then update the recording time info.
        If ZDSoft.SDK.ScnLib_IsRecording() Then
            Dim t As New StringBuilder(11)

            ' Get the current recording time string in the format of 'HH:MM:SS'.
            ZDSoft.SDK.ScnLib_GetRecTimeW(t)

            rec_time.Content = t.ToString()
        Else
            ' If it's not recording then stop recording on failure.
            ' The recording may stop automatically if a fatal error occurs.
            ' In this case you still need to do some cleanup and UI updates.
            Stop_Recording(False)

            Update_UI()
        End If

    End Sub

    Private Sub On_Timer_FPS(sender As Object, e As EventArgs)

        ' Is currently game capture mode and a Direct3D/OpenGL game window detected?
        If ZDSoft.SDK.ScnLib_GetGameWnd() <> IntPtr.Zero Then
            ' Update the game FPS overlay.
            Update_Game_FPS_Overlay()
        End If

    End Sub

    ' Create the game FPS DIB for drawing
    Dim imageFPS As New Bitmap(170, 70, PixelFormat.Format24bppRgb)

    Private Sub Update_Game_FPS_Overlay()

        ' Prepare a graphics object to draw on the game FPS DIB.
        Dim g As Graphics = Graphics.FromImage(imageFPS)

        ' Get the game FPS number.
        Dim fps As Integer = ZDSoft.SDK.ScnLib_GetGameFrameRate()

        ' Place the game FPS number top vertically and right horizontally.
        Dim fpsFormat As New StringFormat()
        fpsFormat.Alignment = StringAlignment.Far
        fpsFormat.LineAlignment = StringAlignment.Near

        Dim t As New StringBuilder(11)

        ' Get the current recording time string in the format of 'HH:MM:SS'.
        ZDSoft.SDK.ScnLib_GetRecTimeW(t)

        ' Place the REC time bottom vertically and right horizontally.
        Dim timeFormat As New StringFormat()
        timeFormat.Alignment = StringAlignment.Far
        timeFormat.LineAlignment = StringAlignment.Far

        ' Clear the overlay background.
        g.FillRectangle(
            Brushes.White,
            New Rectangle(0, 0, imageFPS.Width, imageFPS.Height))

        ' Display a camera icon on the left side.
        g.DrawIcon(System.Drawing.Icon.ExtractAssociatedIcon(
            System.Reflection.Assembly.GetEntryAssembly().ManifestModule.Name),
            3, 3)

        ' Draw the game FPS number.
        g.DrawString(
            fps.ToString() + " fps",
            New Font("Impact", 30, System.Drawing.FontStyle.Regular, GraphicsUnit.Pixel),
            Brushes.Gray,
            New Rectangle(0, 0, imageFPS.Width, imageFPS.Height),
            fpsFormat)

        ' Draw the REC time.
        Dim brushTime As Brush = Brushes.LimeGreen

        If ZDSoft.SDK.ScnLib_IsRecording() Then
            brushTime = Brushes.Red
        End If

        g.DrawString(
            t.ToString(),
            New Font("Impact", 25, System.Drawing.FontStyle.Regular, GraphicsUnit.Pixel),
            brushTime,
            New Rectangle(0, 0, imageFPS.Width, imageFPS.Height),
            timeFormat)

        ' Lock the DIB and get the bits buffer pointer.
        Dim dibFPS As BitmapData = imageFPS.LockBits(
            New Rectangle(0, 0, imageFPS.Width, imageFPS.Height),
            ImageLockMode.ReadOnly,
            PixelFormat.Format24bppRgb)

        ' Show the game FPS overlay on the top-right corner of the game screen.
        ZDSoft.SDK.ScnLib_SetInGameOverlayPosition(ZDSoft.SDK.POSITION_TOP_RIGHT, 5, 5)
        ZDSoft.SDK.ScnLib_ShowInGameOverlay(dibFPS.Scan0, dibFPS.Width, dibFPS.Height, 24)

        ' Don't forget to unlock the DIB.
        imageFPS.UnlockBits(dibFPS)

    End Sub

    Private Sub Update_UI()

        ' Update the radio boxes in the Screen Capture group.
        full_screen.IsEnabled = Not ZDSoft.SDK.ScnLib_IsRecording()
        a_region_window.IsEnabled = Not ZDSoft.SDK.ScnLib_IsRecording()
        pc_game_screen.IsEnabled = Not ZDSoft.SDK.ScnLib_IsRecording()

        ' Update the check boxes in the Audio Capture group.
        playback_audio.IsChecked = ZDSoft.SDK.ScnLib_IsRecordAudioSource(True) ' True - the playback audio source (speakers / headphone)
        microphone_audio.IsChecked = ZDSoft.SDK.ScnLib_IsRecordAudioSource(False) ' False - the recording audio source (microphone / line-In)

        ' Update the check boxes in the Webcam Capture group
        ' The webcam capture Is enabled only when a webcam device Is selected.
        ' The selected webcam device index Is a zero-based number. The first webcam device index Is 0.
        If ZDSoft.SDK.ScnLib_GetSelectedWebcamDevice() >= 0 Then
            webcam_preview.IsEnabled = True

            ' If the webcam Is being previewed you can get a non-zero preview window handle.
            If ZDSoft.SDK.ScnLib_GetWebcamPreviewWnd() <> IntPtr.Zero Then
                webcam_preview.IsChecked = True
            Else
                ' Otherwise the webcam Is Not being previewed.
                webcam_preview.IsChecked = False
            End If

            ' If it's set to record webcam only, screen capture will be discarded and you will get a full webcam video.
            If ZDSoft.SDK.ScnLib_IsRecordWebcamOnly() Then
                full_webcam_video.IsChecked = True
                webcam_overlay.IsChecked = False
            Else
                ' Otherwise you will get a screen recording video with a webcam overlay on it.
                webcam_overlay.IsChecked = True
                full_webcam_video.IsChecked = False
            End If
        Else
            ' A negative index means no webcam device Is selected.
            webcam_preview.IsEnabled = False
            webcam_preview.IsChecked = False
            webcam_overlay.IsChecked = False
            full_webcam_video.IsChecked = False
        End If

        ' Update the elements in the Output Video group.
        Dim videoPath As New StringBuilder(260)

        ' Get the output video file path to be created.
        ZDSoft.SDK.ScnLib_GetVideoPathW(videoPath, False)

        video_path.Text = videoPath.ToString()
        video_path.IsEnabled = Not ZDSoft.SDK.ScnLib_IsRecording()
        change_video_path.IsEnabled = Not ZDSoft.SDK.ScnLib_IsRecording()

        ' Update the recording control buttons at the bottom.
        If ZDSoft.SDK.ScnLib_IsRecording() Then
            start_stop.SetResourceReference(ContentProperty, "stop_str")
        Else
            start_stop.SetResourceReference(ContentProperty, "start_str")
        End If

        If ZDSoft.SDK.ScnLib_IsPaused() Then
            pause_resume.SetResourceReference(ContentProperty, "resume_str")
        Else
            pause_resume.SetResourceReference(ContentProperty, "pause_str")
        End If

        pause_resume.IsEnabled = ZDSoft.SDK.ScnLib_IsRecording()

    End Sub

    Private Sub Start_Recording()

        ' If no foreground Direct3D/OpenGL game window Is detected, prompt user what to do.
        If ZDSoft.SDK.ScnLib_IsGameCaptureModeEnabled() And ZDSoft.SDK.ScnLib_GetGameWnd() = IntPtr.Zero Then
            Dim msgText As String = "No foreground Direct3D/OpenGL game window is detected!" & vbCrLf & vbCrLf &
"Please open a Direct3D/OpenGL game and place its window in the foreground, or the SDK will automatically fallback to standard screen capture mode."
            MessageBox.Show(msgText, Title, MessageBoxButton.OK, MessageBoxImage.Information)
        End If

        ' Set the output video file path to be created.
        ' The file path can include SDK-defined variables: <num>, <date> and <time>
        ZDSoft.SDK.ScnLib_SetVideoPathW(video_path.Text)

        ' Start recording now.
        If ZDSoft.SDK.ScnLib_StartRecording() Then
            ' Start the timer to check the recording status and update recording time.
            timerREC.Start()
        Else
            ' Do some cleanup if recording failed to start.
            Stop_Recording(False)
        End If

    End Sub

    Private Sub Stop_Recording(bSuccessful As Boolean)

        ' Stop the timer
        timerREC.Stop()

        ' Stop the recording. It's OK to call it even if no recording is in progress.
        ZDSoft.SDK.ScnLib_StopRecording()

        Dim filePath As New StringBuilder(260)

        If bSuccessful Then
            ' Get the saved video file path if the recording Is done successfully.
            ZDSoft.SDK.ScnLib_GetVideoPathW(filePath, True)
        Else
            ' Get the saved log file path if the recording Is failed.
            ZDSoft.SDK.ScnLib_GetLogPathW(filePath)
        End If

        ' Play the video file or show the log file.
        If File.Exists(filePath.ToString()) Then
            Dim proc As New Process
            proc.StartInfo.CreateNoWindow = False
            proc.StartInfo.UseShellExecute = True
            proc.StartInfo.FileName = filePath.ToString()
            proc.StartInfo.Verb = "open"
            proc.Start()
            proc.Close()
        End If

    End Sub

    Private Sub start_stop_Click(sender As Object, e As RoutedEventArgs)

        If Not ZDSoft.SDK.ScnLib_IsRecording() Then
            ' Start recording if no recording Is in progress.
            Start_Recording()
        Else
            ' Stop recording if a recording Is in progress.
            Stop_Recording(True)
        End If

        Update_UI()

    End Sub

    Private Sub pause_resume_Click(sender As Object, e As RoutedEventArgs)

        If Not ZDSoft.SDK.ScnLib_IsPaused() Then
            ' Pause recording if the recording Is Not paused.
            ZDSoft.SDK.ScnLib_PauseRecording()
        Else
            ' Resume recording if the recording Is paused.
            ZDSoft.SDK.ScnLib_ResumeRecording()
        End If

        Update_UI()

    End Sub

    Private Sub more_settings_Click(sender As Object, e As RoutedEventArgs)

        Dim hwnd As IntPtr = New WindowInteropHelper(Application.Current.MainWindow).Handle

        ' Popup the SDK built-in settings dialog to let user configure more settings.
        ' If you don't like the style of the SDK built-in settings dialog, you may implement your own.
        ZDSoft.SDK.ScnLib_ConfigureSettings(hwnd)

        Update_UI()

    End Sub

    Private Sub change_video_path_Click(sender As Object, e As RoutedEventArgs)

        Dim dlg As New Microsoft.Win32.SaveFileDialog()

        ' Supported video file formats are MP4, FLV and AVI.
        dlg.Filter = "MP4 videos|*.mp4|FLV videos|*.flv|AVI videos|*.avi"
        dlg.DefaultExt = ".mp4"
        dlg.Title = "Set Output Video File Path"

        ' Popup a save file dialog to let user change the output video file path.
        If dlg.ShowDialog() Then
            ' Set the output video file path to be created.
            ZDSoft.SDK.ScnLib_SetVideoPathW(dlg.FileName)

            Update_UI()
        End If

    End Sub

    Private Sub browse_video_path_Click(sender As Object, e As RoutedEventArgs)

        Dim videoPath As New StringBuilder(260)

        ' Get the output video file path to be created.
        ZDSoft.SDK.ScnLib_GetVideoPathW(videoPath, False)

        ' Remove the invalid file path chars that are used by SDK-defined variables: <num>, <date> and <time>
        videoPath = videoPath.Replace("<", "")
        videoPath = videoPath.Replace(">", "")

        ' Get the folder where your recording videos are saved.
        Dim videoFolder As String = Path.GetDirectoryName(videoPath.ToString())

        If Directory.Exists(videoFolder) Then
            ' Browse the video folder in the Windows File Explorer.
            Process.Start("explorer.exe", videoFolder)
        End If

    End Sub

    Private Sub full_screen_Checked(sender As Object, e As RoutedEventArgs)

        ' Don't bind any capture window.
        ZDSoft.SDK.ScnLib_SetCaptureWnd(IntPtr.Zero, False)
        ' Set all-zero coordinates to let SDK detect and use the full screen coordinates.
        ZDSoft.SDK.ScnLib_SetCaptureRegion(0, 0, 0, 0)
        ' Make sure the game capture mode Is disabled.
        ZDSoft.SDK.ScnLib_EnableGameCaptureMode(False)

    End Sub

    Private Sub a_region_window_Checked(sender As Object, e As RoutedEventArgs)

        Dim l As Int32 = 0
        Dim t As Int32 = 0
        Dim r As Int32 = 0
        Dim b As Int32 = 0
        Dim hwnd As IntPtr = IntPtr.Zero

        ' Turn mouse cursor into a crosshair and let user select a region/window on screen.
        ' To select a region:
        '   1) Move the mouse cursor to the top-left corner of the region you want to capture.
        '   2) Press and hold the left mouse button, and then move the mouse cursor to the bottom-right corner of the region you want to capture.
        '   3) Release the left mouse button, done. Or, you can click the right mouse button to cancel in the middle of the process.
        ' To select a window:
        '   1) Move the mouse cursor over the window you want to capture and then the window will be highlighted.
        '   2) Click the left mouse button to select the pointing window.
        ' To cancel selection, click the right mouse button.
        ' If a region Is selected, you will get the region coordinates and a zero window handle.
        ' If a window Is selected, you will get the window coordinates and a non-zero window handle.
        If ZDSoft.SDK.ScnLib_SelectCaptureRegionW(l, t, r, b, hwnd, "") Then
            ' Bind the capture window if hwnd Is non-zero.
            ' or unbind the capture window if hwnd Is zero.
            ZDSoft.SDK.ScnLib_SetCaptureWnd(hwnd, True)
            ' Set the capture region coordinates.
            ZDSoft.SDK.ScnLib_SetCaptureRegion(l, t, r, b)
        End If

        ' Make sure the game capture mode Is disabled.
        ZDSoft.SDK.ScnLib_EnableGameCaptureMode(False)

    End Sub

    Private Sub pc_game_screen_Checked(sender As Object, e As RoutedEventArgs)

        ' Don't bind any capture window. (You can bind a specific game window if you don't want SDK to detect foreground game window automatically)
        ZDSoft.SDK.ScnLib_SetCaptureWnd(IntPtr.Zero, False)
        ' Set all-zero coordinates to let SDK detect and use the full screen coordinates if no game screen Is detected.
        ZDSoft.SDK.ScnLib_SetCaptureRegion(0, 0, 0, 0)
        ' Enable the game capture mode.
        ZDSoft.SDK.ScnLib_EnableGameCaptureMode(True)

    End Sub

    Private Sub playback_audio_Checked(sender As Object, e As RoutedEventArgs)

        ' Enable capture from the playback audio source (speakers / headphone).
        ZDSoft.SDK.ScnLib_RecordAudioSource(True, True)

    End Sub

    Private Sub playback_audio_Unchecked(sender As Object, e As RoutedEventArgs)

        ' Disable capture from the playback audio source (speakers / headphone).
        ZDSoft.SDK.ScnLib_RecordAudioSource(True, False)

    End Sub

    Private Sub microphone_audio_Checked(sender As Object, e As RoutedEventArgs)

        ' Enable capture from the recording audio source (microphone / line-in).
        ZDSoft.SDK.ScnLib_RecordAudioSource(False, True)

    End Sub

    Private Sub microphone_audio_Unchecked(sender As Object, e As RoutedEventArgs)

        ' Disable capture from the recording audio source (microphone / line-in).
        ZDSoft.SDK.ScnLib_RecordAudioSource(False, False)

    End Sub

    Private Sub webcam_preview_Checked(sender As Object, e As RoutedEventArgs)

        ' Open the webcam preview window.
        ZDSoft.SDK.ScnLib_PreviewWebcam(True, IntPtr.Zero, True, 0)

    End Sub

    Private Sub webcam_preview_Unchecked(sender As Object, e As RoutedEventArgs)

        ' Close the webcam preview window.
        ZDSoft.SDK.ScnLib_PreviewWebcam(False, IntPtr.Zero, True, 0)

    End Sub

    Private Sub webcam_overlay_Checked(sender As Object, e As RoutedEventArgs)

        ' Enable webcam capture by selecting the first webcam device.
        ZDSoft.SDK.ScnLib_SelectWebcamDevice(0)
        ' Don't record webcam only. Put the webcam overlay on the screen capture video.
        ZDSoft.SDK.ScnLib_RecordWebcamOnly(False)

        Update_UI()

    End Sub

    Private Sub webcam_overlay_Unchecked(sender As Object, e As RoutedEventArgs)

        If Not full_webcam_video.IsChecked Then
            ' Close the webcam preview window.
            ZDSoft.SDK.ScnLib_PreviewWebcam(False, IntPtr.Zero, True, 0)
            ' Disable webcam capture by selecting a negative index.
            ZDSoft.SDK.ScnLib_SelectWebcamDevice(-1)
        End If

        Update_UI()

    End Sub

    Private Sub full_webcam_video_Checked(sender As Object, e As RoutedEventArgs)

        ' Enable webcam capture by selecting the first webcam device.
        ZDSoft.SDK.ScnLib_SelectWebcamDevice(0)
        ' Record webcam only. Discard screen capture. Make a full webcam video.
        ZDSoft.SDK.ScnLib_RecordWebcamOnly(True)

        Update_UI()

    End Sub

    Private Sub full_webcam_video_Unchecked(sender As Object, e As RoutedEventArgs)

        If Not webcam_overlay.IsChecked Then
            ' Close the webcam preview window.
            ZDSoft.SDK.ScnLib_PreviewWebcam(False, IntPtr.Zero, True, 0)
            ' Disable webcam capture by selecting a negative index.
            ZDSoft.SDK.ScnLib_SelectWebcamDevice(-1)
        End If

        Update_UI()

    End Sub
End Class
